'use strict';

document.addEventListener("DOMContentLoaded", function () {
	// Удаляем ключ image из объектов
	function removeImageKey(array) {
		return array.map(({ image, ...rest }) => rest);
	}

	// Получение актуальных товаров из localStorage без ключа image
	function getItemsWithoutImages() {
		const cart = JSON.parse(localStorage.getItem("cart")) || [];
		return removeImageKey(cart);
	}

	const clientEmail = String(localStorage.getItem("clientEmail")) || '';

	const iawDefault = {
		merchant: 'WIDGET',
		btnId: 'iawBtn',
		customer: {
			email: "customer@mail.ru"
		},
		bundle: {
			fiscalInfo: {
				clientEmail
			}
		}
	};

	class vtbIAW {
		constructor(options) {
			this.iawClose = document.createElement('div');
			this.frameWrapper = document.createElement('div');
			this.iawCloseBtnWrapper = document.createElement('div');
			this.iawCloseBtn = document.createElement('button');
			this.iawFrame = document.createElement('iframe');
			this.iawStyle = document.createElement('link');

			this.iawBtn = options.btnId;
			this.iawUrl = 'https://pay.vtb.ru/api/tsp-ia-box-cloud-cms/v1/orders';
			this.iawMerchant = options.merchant;
			this.iawCustomer = options.customer;
			this.iawFiscalInfo = options.bundle?.fiscalInfo || {};

			if (!this.iawMerchant) throw new Error('[IAW] Credential: Merchant lost');
			if (!this.iawBtn) throw new Error('[IAW] Button ID cannot be empty');

			this.#init();
		}

		#getAttr(target) {
			return {
				price: target.attributes['data-price'].value,
				id: this.#getUid()
			};
		}

		#generateRandomCode() {
			let code = '';
			for (let i = 0; i < 4; i++) {
				code += Math.floor(Math.random() * 10);
			}
			return code;
		}

		#getUid() {
			const date = new Date();
			const y = date.getFullYear();
			const m = String(date.getMonth() + 1).padStart(2, '0');
			const d = String(date.getDate()).padStart(2, '0');
			const h = String(date.getHours()).padStart(2, '0');
			const min = String(date.getMinutes()).padStart(2, '0');
			const s = String(date.getSeconds()).padStart(2, '0');
			const ms = String(date.getMilliseconds()).padStart(3, '0');
			return `WT${this.#generateRandomCode()}${y}${m}${d}${h}${min}${s}${ms}${this.#generateRandomCode()}`;
		}

		#setStyle() {
			this.iawStyle.setAttribute("href", "./public/css/iaw.css");
			this.iawStyle.setAttribute("rel", "stylesheet");
			document.head.appendChild(this.iawStyle);
		}

		#setFrame(url) {
			if (!url) throw new Error('[IAW] Empty URL for frame');

			this.iawFrame.setAttribute('src', url);
			this.iawFrame.setAttribute('class', 'iawFrame');
			this.iawFrame.setAttribute('allowpaymentrequest', 'true');
			this.iawFrame.setAttribute('allowtransparency', 'true');
			this.iawFrame.setAttribute('frameborder', '0');
			this.iawFrame.setAttribute('loading', 'lazy');

			this.frameWrapper.className = 'frameWrapper';
			this.#setClose();

			this.frameWrapper.appendChild(this.iawFrame);
			this.iawClose.appendChild(this.frameWrapper);
			this.iawClose.setAttribute('class', 'iawShow');
		}

		#setClose() {
			this.iawCloseBtn.setAttribute('type', 'button');
			this.iawCloseBtn.setAttribute('class', 'iawCloseButton');
			this.iawCloseBtnWrapper.setAttribute('class', 'iawCloseBtnWrapper');

			this.iawCloseBtnWrapper.appendChild(this.iawCloseBtn);
			this.frameWrapper.appendChild(this.iawCloseBtnWrapper);
			document.body.appendChild(this.iawClose);
		}

		#setListener() {
			const button = document.querySelector(`#${this.iawBtn}`);
			const _this = this;

			this.iawCloseBtnWrapper.addEventListener('click', (e) => {
				e.preventDefault();
				e.stopPropagation();
				_this.iawClose.remove();
			});

			button.addEventListener('click', async function (e) {
				e.preventDefault();
				e.stopPropagation();

				const payData = _this.#getAttr(e.target);
				await _this.#sendAction(payData);
			});
		}

		async #sendAction(options) {
			let price = options.price.replace(',', '.');
			const timestamp = new Date();
			timestamp.setDate(timestamp.getDate() + 1);

			const args = {
				orderId: options.id,
				orderName: `№ ${options.id}`,
				expire: timestamp.toISOString(),
				amount: {
					value: parseFloat(price),
					code: 'RUB'
				},
				customer: this.iawCustomer,
				bundle: {
					fiscalInfo: this.iawFiscalInfo,
					items: getItemsWithoutImages()
				}
			};

			try {
				const data = await this.#sendRequest('post', this.iawUrl, args);
				if (data?.object?.payUrl) {
					this.#setFrame(data.object.payUrl);
				} else {
					throw new Error('[IAW] No payUrl returned');
				}
			} catch (error) {
				throw new Error(`[IAW] Request failed: ${error}`);
			}
		}

		async #sendRequest(method, url, body = null) {
			const options = {
				method: method.toUpperCase(),
				headers: {
					'Content-Type': 'application/json',
					'Merchant-Host': '',
					'Merchant-Authorization': this.iawMerchant
				},
				redirect: 'follow',
				body: body ? JSON.stringify(body) : null
			};

			const result = await fetch(url, options);
			return result.json();
		}

		#init() {
			this.#setStyle();
			this.#setListener();
		}
	}

	new vtbIAW((typeof iawOptions !== 'undefined') ? iawOptions : iawDefault);
});
